#!/usr/bin/perl
# IBM_PROLOG_BEGIN_TAG 
# This is an automatically generated prolog. 
#  
#  
#  
# Licensed Materials - Property of IBM 
#  
# (C) COPYRIGHT International Business Machines Corp. 2001,2002 
# All Rights Reserved 
#  
# US Government Users Restricted Rights - Use, duplication or 
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp. 
#  
# IBM_PROLOG_END_TAG 

#
#####################################################################
#                                                                   #
# Module: InstallUtils.pm                                           #
#                                                                   #
#-------------------------------------------------------------------#
#                                                                   #
# Description: This module contains Perl subroutines used           #
#              commonly across the CSM installation commands.       #
#              Detailed descriptions appear in subroutine headers.  #
#                                                                   #
# Subroutines:                                                      #
#								    #	  
#  make_node_list_file invokeLAP				    #
#  get_target_nodes get_common_attrs findImagesInPath               #
#  make_config_file create_nodemap_file start_nfs export_csminstall #
#  unexport_csminstall						    #
#  get_lock   release_lock  backup_logfile start_logging            #
#  stop_logging  append_logging  get_management_server  get_OSName  #
#  get_DistributionVersion                                          #
#  get_CSMVersion  get_PkgArchitecture  get_pkgVersion              #
#  get_DistributionName get_uniqu_arry_elemnts                      #
#  findRpmInPath findRpm installRequiredRpms getRequiredRpms        #
#  getReqsNotSatisfied parseRequire inc_ip                          #
#  SetAllowRequest SetMode genHex verify_tftp verify_dhcp           #
#  setup_atftp testvalidconsole testvalidpowermethod chkethup       #
#  getRemoteShell checkHOMEforSSHperm installSSHFiles cpSSHFiles    #
#  setupAIXRSH setupLinuxRSH setupRemoteShellforFullInstall	    #
#  getRecentVersionRpm subnet getUpdateDir getRPMSDir getBaseDir    #
#                                                                   #
#####################################################################
# @(#)91   1.149   src/csm/install/InstallUtils.pm.perl, setup, csm_rpyxh, rpyxht1f3 9/16/02 11:31:45

package InstallUtils;

use strict;
use Fcntl qw(:flock);

#
# Subroutines
#

###########################################################
# invokeLAP
#  To invoke the LAP tool and display the license
#  agreements for CSM.
#
# input: a string - either: "tab" or "full"
# output: 3, 9, or 127
#  The LAP tool will be invoked and copy the LA and LI
#  files to /var/opt/csm/<VRnumber>/<type>/license/<lang>.txt
#  where <VRnumber> = CSM Version (1.3, etc)
#  <type> = tryandbuy or full, depending upon the license.
#  <lang> = one of the 17 supported LAP languages.
#  If everything ran successful, then LAP will return 9.
#  otherwise, if the user returns 3, the license was not 
#  accepted.  If something went wrong with the LAP tool, then
#  an error of 127 will be returned.
###########################################################
sub invokeLAP{
    my ($class, $licType) = @_; 
    my ($csmVersion,$laDir,$strCmd, $rc); 
    my $laptool = "/opt/csm/install/lapapp.jar";
    # make sure that $licType is either tab or full
    if($licType eq "full"){
	# do nothing.. everything's fine.
    }elsif($licType eq "tab"){
	$licType = "tryandbuy";
    }else{
	print "don't know..\n";
	return 127;
    }  

    # get the CSM version:
    $csmVersion = InstallUtils->get_CSMVersion("csm.server"); 
    $csmVersion = sprintf "%.1f", $csmVersion;    # format to be
						  # x.x, e.g.: 1.3

    # check that the LA and LI files are in place:
    if(NodeUtils->isLinux()){
        $laDir = "/opt/csm/install/$csmVersion/Linux/license/$licType";
    }elsif(NodeUtils->isAIX()){
        $laDir = "/opt/csm/install/$csmVersion/AIX/license/$licType";
    }else{
	return 127;
    }

    unless( -f "$laDir/LA_en"){
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,
               'csminstall','E', 'EMsgCANT_READ_FILE', "$laDir/LA_en"); 
	return 127;   
    }

    # check that the LAP tool is in place:
    unless ( -f "$laptool"){
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,
                'csminstall','E', 'EMsgCANT_READ_FILE', $laptool); 
	return 127;  
    }
	
    # Everything should be in place now, so let's call LAP:
    # construct the command:
    if(NodeUtils->isLinux()){
	# check that Java is installed... and use the right one.
	if(-d "/opt/IBMJava2-131"){
            $strCmd = "/opt/IBMJava2-131/jre/bin/java -cp ";
  	    $strCmd .= "/opt/csm/install/lapapp.jar:";
	    $strCmd .= "/opt/IBMJava2-131/jre/lib/rt.jar:";
	    $strCmd .= "/opt/IBMJava2-131/jre/lib/i18n.jar:";
	}elsif(-d "/opt/IBMJava2-13"){
            $strCmd = "/opt/IBMJava2-13/jre/bin/java -cp ";
  	    $strCmd .= "/opt/csm/install/lapapp.jar:";
	    $strCmd .= "/opt/IBMJava2-13/jre/lib/rt.jar:";
	    $strCmd .= "/opt/IBMJava2-13/jre/lib/i18n.jar:";
	}else{
	    print "IBMJava2-13[1] is not installed.\n";
	    exit 127;
	}
	
    }else{ # AIX
        $strCmd = "/usr/java131/jre/bin/java -cp ";
  	$strCmd .= "/opt/csm/install/lapapp.jar:";
	$strCmd .= "/usr/java131/jre/lib/rt.jar:";
	$strCmd .= "/usr/java131/jre/lib/i18n.jar:";
    }
     
    # this part of the command is common in both AIX and Linux:
    $strCmd .= " com.ibm.lex.lapapp.LAP ";
    $strCmd .= "-l $laDir ";
    $strCmd .= "-s /var/opt/csm/$csmVersion/$licType ";
    $strCmd .= "-text_only";
    $rc = system("$strCmd");
    $rc = $rc >> 8;
    return $rc;
}


###########################################################
# make_node_list_file
#  this file is suitable for the ENV vars: 
#	DSH_LIST
#	RPOWER_LIST
#	RCONSOLE_LIST
#
#  IMPORTANT!!!: if you use this subroutine, make sure you use 
#  	InstallUtils->close_delete_file($file_handle, $file_name) 
#       to erase it.
#
# input: (\@list_of_nodes)
# output
#	$file_name
# sets global var: $::NODE_LIST_FILE
###########################################################
sub make_node_list_file{
	my ($class, $ref_node_list) = @_;
	my @node_list = @$ref_node_list;
	srand(time|$$); #random number generator start
	
	my $file = "/tmp/$$";
        while(-e $file){
             $file = InstallUtils->CreateRandomName($file);
        }

	open($::NODE_LIST_FILE, ">$file") or NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgCANT_WRITE_FILE', "$file");
	foreach my $node (@node_list){
		print $::NODE_LIST_FILE "$node\n";
	}
	return $file;
}
###########################################################
# close_delete_file
# This subroutine takes a file handle and file name
# First closes and then deletes the file
# input: $file_handle $file_name
# output null
###########################################################
sub close_delete_file{
	my ($class, $file_handle, $file_name) = @_;
	close $file_handle;
	
  	my $cmd = "$::RM -f $file_name";
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
        my $rc  = system ($cmd) >> 8;
        if ($rc) {
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
        }
}
###########################################################
# CreateRandomName
# input : prefix of name
# output: prefix with 8 random letters appended
###########################################################
sub CreateRandomName {

    my ($class, $name) = @_;

    my $nI;
    for ($nI = 0; $nI < 8; $nI++) {
       my $char = ('a'..'z', 'A'..'Z')[int(rand( 52)) + 1];
       $name .= $char;
    }
    $name;
}
###########################################################
# get_target_nodes
#
# input:
#      (\@ARGV) either from @ARGV or $::opt_n
#      ($node group string) from $::opt_N
#	$file containing node list
#
#Global variables:
#	$::ALL                     	#Distribute to all nodes
#	$::MANAGED     	#Distribute to all nodes where Mode="Managed"
#       $::PRE_MANAGED	#Distribute to all nodes where Mode="PreManaged"
#
# output
#	\@DestNode  = reference to list of all target nodes
#       \@lsnode_info = reference to a list of results from lsnode command (only for the nodes in @DestNode)
#	\%DestNodeHash = reference to a hash of hostnames and attributes (foreach node in @Destnode):
#			$DestNodeHash{$hostname}{$attribute_name} = "attribute value"
#			For the list of attributes see $lsnode_attr
###########################################################
sub get_target_nodes{
    my ($class, $ref_nodelist, $nodegrp_list, $file) = @_;
    my @nodelist = @$ref_nodelist;
    my (@DestNode, @DestGroup, %DestKeyTemp, %all_groups, @lsnode_info, @empty_array);
    
    my @real_nodelist; #split the nodes if there are commas
        foreach my $n (@nodelist){
    	   if($n =~ m/,/){
    		   my @tmp = split ',', $n;
    		   foreach my $t (@tmp){
    			   push @real_nodelist, $t;
    		   }
    	   }
    	   else{
    		   push @real_nodelist, $n;
    	   }
       }
       @nodelist=@real_nodelist;
   undef @real_nodelist;
    
    my $lsnode_attr = "Hostname,ManagementServer,InstallOSName,InstallCSMVersion,InstallDistributionVersion,InstallDistributionName,InstallPkgArchitecture,Mode,InstallAdapterMacaddr,InstallAdapterType,InstallMethod";
    # These delimeters are used with the lsnode command.  A simple colon (:)
    # cannot be used since there are colons in InstallAdapterMacaddr.
    #my $delim = ':|:';		# Used on the lsnode command
    my $check_delim = ':\|:';	# Used when splitting the line apart
    
    if($::ALL || $::MANAGED || $::PRE_MANAGED ){ #distribute to all nodes of a certain set
        my $select_string;
        if($::MANAGED){
            $select_string = "Mode=\"Managed\"";
        } elsif($::PRE_MANAGED){
            $select_string = "Mode=\"PreManaged\"";
        } else{ #just managednodes
            $select_string = "";
        }
        my %opthash = (WhereStr => $select_string); 
	my $ref_lsnode_info = NodeUtils->listNodeAttrs( \@empty_array, $lsnode_attr, \%opthash);
	@lsnode_info = @$ref_lsnode_info;
        
        #foreach my $line (@lsnode_info){
        #    chomp $line;
        #    my $hostname = (split /$check_delim/, $line)[0];
        #    push @DestNode, $hostname;
        #}
    } 
    else{ #we are not distributing to all nodes of a certain mode 
        my %opthashref = ();
    	if($file){
		%opthashref = ( NodeFile => $file, NodeGrp => $nodegrp_list);
	}
	else{
		%opthashref = ( NodeFile => 1, NodeGrp => $nodegrp_list);
	}	
	my $refnodes = NodeUtils->gatherNodeList(\@nodelist, \%opthashref);
	if(scalar @$refnodes){
		my $ref_lsnode_info = NodeUtils->listNodeAttrs( $refnodes, $lsnode_attr);
		@lsnode_info = @$ref_lsnode_info;
	}
	else{
		@lsnode_info = ();
	}
    }
    
    my  %DestNodeHash;
    foreach my $line (@lsnode_info){  #this only has node names in it from the cluster
	my @attributes = split /$check_delim/, $line;
	my $hostname = $attributes[0];
	chomp $line;
	my @attr_names = split ',', $lsnode_attr;
	my $index =0;
	foreach my $attr (@attr_names){
	    $DestNodeHash{$hostname}{$attr} = $attributes[$index];
	    $index++;
	}
    }
    
    if(@::VALID_INSTALLMETHODS){
	my %meths;
	foreach my $method (@::VALID_INSTALLMETHODS){
	    $meths{$method}=1;
	}
	foreach my $target (keys %DestNodeHash){
	    if(!$meths{$DestNodeHash{$target}{'InstallMethod'}}){
		if(!($::ALL || $::MANAGED || $::PRE_MANAGED)){                  
		    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E1', 'EMsgINVALID_INSTALL_METH', $target, (keys %meths));
		}
		else{
			NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','W', 'EMsgINVALID_INSTALL_METH', $target, (keys %meths));
			delete $DestNodeHash{$target};
		    }
	    }
	}
    }

    my @tmpDestNode = keys %DestNodeHash;
    my @return_array = (\@tmpDestNode, \@lsnode_info, \%DestNodeHash);
    return @return_array;
}

# get_common_attrs
#
# Get the common attributes across all the nodes.
#  InstallDistributionName
#  InstallDistributionVersion
#  InstallPkgArchitecture
#  InstallCSMVersion
#
# input:
#      none
# global varialbles:
#      %::NODEHASH - List of nodes to check
# output:
#      The following variables in an array:
#      ($distro_name, $distro_version, $arch, $csm_version);
#
# Exits with an error if one of the attributes is not the same on 
# all the nodes, 
#
sub get_common_attrs
{
    my ($hostname, $distro_name, $distro_version, $arch, $csm_version);
    my ($errors) = 0;
    my ($InstallDistributionName);
    my ($InstallDistributionVersion);
    my ($InstallPkgArchitecture);
    my ($InstallCSMVersion);

    foreach $hostname (keys %::NODEHASH)
    {
	$InstallDistributionName = 
	    $::NODEHASH{$hostname}{'InstallDistributionName'};
	$InstallDistributionVersion = 
	    $::NODEHASH{$hostname}{'InstallDistributionVersion'};
	$InstallPkgArchitecture = 
	    $::NODEHASH{$hostname}{'InstallPkgArchitecture'};
	$InstallCSMVersion = 
	    $::NODEHASH{$hostname}{'InstallCSMVersion'};

	if (! $distro_name)
	{
	    $distro_name = $InstallDistributionName;
	}
	if (! $distro_version)
	{
	    $distro_version = $InstallDistributionVersion;
	}
	if (! $arch)
	{
	    $arch = $InstallPkgArchitecture;
	}
	if (! $csm_version)
	{
	    $csm_version = $InstallCSMVersion;
	}

	if ("$distro_name" and 
	    ($InstallDistributionName ne $distro_name))
	{
	    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgDIFFERENT_ATTR', 'InstallDistributionName');
	    $errors++;
	}

	if ("$distro_version" and 
	    ($InstallDistributionVersion ne $distro_version))
	{
	    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgDIFFERENT_ATTR', 'InstallDistributionVersion');
	    $errors++;
	}

	if ("$arch" and 
	    ($InstallPkgArchitecture ne $arch))
	{
	    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgDIFFERENT_ATTR', 'InstallPkgArchitecture');
	    $errors++;
	}

	if ("$csm_version" and 
	    ($InstallCSMVersion ne $csm_version))
	{
	    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgDIFFERENT_ATTR', 'InstallCSMVersion');
	    $errors++;
	}
    }

    if (! $distro_name) {
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgNO_ATTR', 'InstallDistributionName');
	$errors++;
    }
    if (! $distro_version) {
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgNO_ATTR', 'InstallDistributionVersion');
	$errors++;
    }
    if (! $arch) {
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgNO_ATTR', 'InstallPkgArchitecture');
	$errors++;
    }
    if (! $csm_version) {
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgNO_ATTR', 'InstallCSMVersion');
	$errors++;
    }


    exit $errors if ($errors);

    my @return_array = ($distro_name, $distro_version, $arch, $csm_version);
    return (@return_array);
}

#-------------------------------------------------------------------------#
#                                                                         #
#  SetAllowRequest     Make sure the management server can update the     #
#                      node information. 				  #	
#		       	This should be set to 3 by installnode		  #
#		 	and by updatenode if the Mode is not managed.	  #
#			Otherwise it should be set to 2 by updatenode.    #
#			during updatenode exit, it should be set to 0.    #
#                                                                         #
#   Return Codes: 0 - All was successful.                                 #
#                 1 - An error occured.                                   #
#-------------------------------------------------------------------------#
sub SetAllowRequest {

    my ($class, $ref_DestNodeHash, $value) = @_;
    my %DestNodeHash = %$ref_DestNodeHash;
    if(!$value){
	    $value = "0 ";
    }
    if(keys %DestNodeHash){
	    # DestNodeHash is a hash containing attribute values for 
	    # each node that was provided on the command line.
	    my ( $cmd, $rc);

	    # check each node on our list and set the AllowManageRequest 
	    # attribute if needed.
	    my @nodelist = (keys %DestNodeHash);
	    
	    my $nodelist_file = 
		InstallUtils->make_node_list_file(\@nodelist);
	    $cmd = "$::CHNODE -f $nodelist_file AllowManageRequest=$value";
	    $rc = system("$cmd");
	    if ($rc >> 8){
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCMD_FAILED_RC', "$cmd", $rc);
	    }
	    InstallUtils->close_delete_file($::NODE_LIST_FILE, $nodelist_file);
	    return $::OK;
	    
    }
    else{
	    return 1; #a value and nodes must be specified.
    }
}

# Set a node's Mode attribute to PreManaged, Installing or Managed.
sub SetMode
{
    my ($class, $node, $mode) = @_;
    my ($cmd, $rc, $fstatus);

    if (($mode ne "PreManaged") and 
	($mode ne "Installing") and 
	($mode ne "Managed"))
    {
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E1', 'EMsgINVALID_MODE', $mode, "PreManaged, Installing, Managed");
    }

    $cmd = "$::CHNODE $node Mode=$mode";
    $rc = system($cmd) >> 8;
    if ($rc != 0)
    {
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCMD_FAILED_RC', $::CHNODE, $rc);
	$fstatus = $rc;
    }

    return $fstatus;
}

#-------------------------------------------------------------------------#
#                                                                         #
#  make_config_file  - Create configuration information files for each    #
#                      node provided in the list.                         #
#                                                                         #
#	 requires $::REMOTE_SHELL and $::SETUP_REMOTE_SHELL to be set     #
#                                                                         #
#   Return Codes: 0 - All was successful.                                 #
#                 1 - An error occured.                                   #
#-------------------------------------------------------------------------#
sub make_config_file {

    my ($class, $ref_nl, $ref_nodeinfo, $ref_DestNodeHash) = @_;
	my @nl = @$ref_nl;
	my @nodeinfo = @$ref_nodeinfo;

    #  DestNodeHash is a hash containing attribute values for each node
    #    that was provided on the command line.
	my %DestNodeHash = %$ref_DestNodeHash;

    my ($hname, $rc);
    my $csmcfgdir = "/csminstall/csm/config";
    my $cmd;
    my $output;
    my $DirectorInstalled=0;

    #
    #  Check and/or create the /csminstall/csm/config directory
    #
    if (! -d $csmcfgdir) {
        $cmd = "$::MKDIR -m 755 -p $csmcfgdir";
        $rc = system("$cmd");
        if ($rc >> 8) {
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgNO_CreateDir', $csmcfgdir);
            return($::NOK);
        }
    }

    # If this is a Linux management server, check if csm.director.agent 
    # is installed.
    if(NodeUtils->isLinux()==1) {
        $cmd = "$::RPMCMD -q csm.director.agent";
        $output = NodeUtils->runcmd($cmd, -1);
	if ($::RUNCMD_RC == 0) {
	    $DirectorInstalled = 1;
	}
    }

    #
    # create the config_info file
    #

    #  go through each node in the list
    #   (the keys are the hostnames as returned from tryHost() !!)
    foreach $hname (keys %DestNodeHash) { 
        
        # remove the old config file if any
        if ( -e "$csmcfgdir/$hname.config_info") {
            $cmd = "$::RM $csmcfgdir/$hname.config_info";
            $rc = system("$cmd");
            if ($rc >> 8){
                my $fn = "$csmcfgdir/$hname.config_info";
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgNO_RM', "$fn");
                 # try to continue ??return $::NOK;
            }
        }

        # create the new file
        unless (open(CONFIG_OUT,">$csmcfgdir/$hname.config_info")) {
            my $fn = "$csmcfgdir/$hname.config_info";
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgCANT_WRITE_FILE', "$fn");
            return $::NOK;
        }

	# Add cluster-wide configuration information to the file
	if ( defined($::REMOTE_SHELL) ) {
            print CONFIG_OUT "RemoteShell=$::REMOTE_SHELL\n";
	}
	if ( defined($::SETUP_REMOTE_SHELL) ) {
            print CONFIG_OUT "SetupRemoteShell=$::SETUP_REMOTE_SHELL\n";
	}

        # add the node configuration information to the file
        if ( defined($hname) ) {
            print CONFIG_OUT "Hostname=$hname\n";
        }
        if ( defined($DestNodeHash{$hname}{'ManagementServer'}) ) {
            print CONFIG_OUT "ManagementServer=$DestNodeHash{$hname}{'ManagementServer'}\n";
        }
        if ( defined($DestNodeHash{$hname}{'InstallOSName'} ) ) {
            print CONFIG_OUT "InstallOSName=$DestNodeHash{$hname}{'InstallOSName'}\n";
        }
        if ( defined($DestNodeHash{$hname}{'InstallCSMVersion'}) ) {
            print CONFIG_OUT "InstallCSMVersion=$DestNodeHash{$hname}{'InstallCSMVersion'}\n";
        }
        if ( defined($DestNodeHash{$hname}{'InstallDistributionVersion'}) ) {
            print CONFIG_OUT "InstallDistributionVersion=$DestNodeHash{$hname}{'InstallDistributionVersion'}\n";
        }
        if ( defined($DestNodeHash{$hname}{'InstallDistributionName'} ) ) {
            print CONFIG_OUT "InstallDistributionName=$DestNodeHash{$hname}{'InstallDistributionName'}\n";
        }
        if ( defined($DestNodeHash{$hname}{'InstallPkgArchitecture'}) ) {
            print CONFIG_OUT "InstallPkgArchitecture=$DestNodeHash{$hname}{'InstallPkgArchitecture'}\n";
        }
        if ( defined($DestNodeHash{$hname}{'Mode'}) ) {
	    print CONFIG_OUT "Mode=$DestNodeHash{$hname}{'Mode'}\n";
        }

	# Handle Linux-specific attributes.
	if ( $DestNodeHash{$hname}{'InstallOSName'}eq "Linux") {

	    # If csm.director.agent is installed on the management server,
	    # then install it on the Linux nodes.
	    print CONFIG_OUT "InstallDirector=$DirectorInstalled\n";
	}

        close(CONFIG_OUT);
        
        $cmd = "$::CHMOD 644 $csmcfgdir/$hname.config_info";
        $rc = system("$cmd");
        if ($rc >> 8){
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCMD_FAILED_RC', $::CHMOD, $rc);
            # try to continue
        }
    }  # go do the next node

    return $::OK;
}

#-------------------------------------------------------------------------#
#    create_nodemap_file                                                  #
#                                                                         #
# Create /csminstall/csm/config/nodemap file that contains a mapping of   #
# ManagedNode hostnames as defined in the CSM database, and node          #
# hostnames as returned by the hostname command.                          #
#                                                                         #
# File format:                                                            #
#      csm_hostname<space>node_hostname                                   #
#                                                                         #
#                                                                         #
#   Return Codes: 0 - All was successful.                                 #
#                 1 - An error occured.                                   #
#-------------------------------------------------------------------------#
sub create_nodemap_file {
    my ($class, $ref_nodelist, $ref_nodeinfo, $ref_DestNodeHash) = @_;
    my @nodelist = @$ref_nodelist;
    my @nodeinfo = @$ref_nodeinfo;
    #  DestNodeHash is a hash containing attribute values for each node
    #    that was provided on the command line.
    my %DestNodeHash = %$ref_DestNodeHash;
    %main::dsh_failed = ();
    my ($cmd, @dshout, $rc);
    my ($csm_nodename, $csmnode, $hostname);
    my $csmcfgdir = "/csminstall/csm/config";
    my $nodemap_file = "/csminstall/csm/config/nodemap";
    my $node;
    my $dshresult="";
    #
    #  Check and/or create the /csminstall/csm/config directory
    #
    if (! -d $csmcfgdir) {
        $cmd = "$::MKDIR -m 755 -p $csmcfgdir";
        $rc = system("$cmd");
        if ($rc >> 8) {
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgNO_CreateDir', $csmcfgdir);
            return($::NOK);
        }
    }
    # Remove any old occurences of the nodemap file
    if (-f $nodemap_file) {
        $cmd = "$::RM $nodemap_file";
        $rc = system("$cmd") >> 8;
        if ($rc != 0) {
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgNO_RM', "$nodemap_file");
            # try to continue
        }
    }
    # open the map file
    unless (open(NODEMAP,">$nodemap_file")) {
        # print error message
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgCANT_WRITE_FILE', "$nodemap_file");
        return $::NOK;
    }
    my @AIX;
    my @Linux;
    my %found = ();
    my %dups = ();
    # for each node in the node list
    foreach $node (keys %DestNodeHash) {
	# set the correct command based on the InstallOSName
        if ( defined($DestNodeHash{$node}{'InstallOSName'}) ) {
            if ($DestNodeHash{$node}{'InstallOSName'} eq "AIX") {
                #$dshcmd = "$::DSH -n$node hostname";
                push @AIX, $node;
            } elsif ($DestNodeHash{$node}{'InstallOSName'} eq "Linux") {
                #$dshcmd = "$::DSH -n$node hostname -f";
                push @Linux, $node;
            }
	} 
    }
    if(@Linux){
	my $node_file = InstallUtils->make_node_list_file(\@Linux); #put all dest nodes in tmp file
	$ENV{'DSH_LIST'} = $node_file;
	my $dshcmd = "$::DSH hostname -f";
	my @dshresult = `$dshcmd`;
	$rc = $? >> 8;
	InstallUtils->close_delete_file($::NODE_LIST_FILE, $node_file); #delete node file
	if ($rc != 0) {
	    # print error message
	    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCMD_FAILED_RC', $dshcmd, $rc);
	    #$main::dsh_failed{$node}=1; #take these out of target nodes
	    #next;
        }
	foreach my $line (@dshresult){
	    my ($csmnode, $hostname) = split(" ",$line);
	    $hostname     =~ s/://g;
	    $csmnode      =~ s/://g;
	    if ( defined($csmnode) && defined($hostname) ) {
		# make sure the format is consistent with the name that
		#     was used to create the configinfo file
		#$csm_nodename = NodeUtils->tryHost($csmnode);
		$found{$csmnode}=1;
		$dups{$hostname}{$csmnode}=1;
		# put the result in the nodemap file
		unless(print NODEMAP "$csmnode $hostname\n")  {
		    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_WRITEtoFILE', "$nodemap_file");
		}
	    }
	}
    }
    if(@AIX){
	my $node_file = InstallUtils->make_node_list_file(\@AIX); #put all dest nodes in tmp file
	$ENV{'DSH_LIST'} = $node_file;
	my $dshcmd = "$::DSH hostname";
	my @dshresult = `$dshcmd`;
	$rc = $? >> 8;
	InstallUtils->close_delete_file($::NODE_LIST_FILE, $node_file); #delete node file
	if ($rc != 0) {
	    # print error message
	    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCMD_FAILED_RC', $dshcmd, $rc);
	    #$main::dsh_failed{$node}=1; #take these out of target nodes
	    #next;
	}
	foreach my $line (@dshresult){
	    my ($csmnode, $hostname) = split(" ",$line);
	    $hostname     =~ s/://g;
	    $csmnode      =~ s/://g;
	    if ( defined($csmnode) && defined($hostname) ) {
		# make sure the format is consistent with the name that
		#     was used to create the configinfo file
		#$csm_nodename = NodeUtils->tryHost($csmnode);
		$found{$csmnode}=1;
		$dups{$hostname}{$csmnode}=1;
		# put the result in the nodemap file
		unless(print NODEMAP "$csmnode $hostname\n")  {
		    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_WRITEtoFILE', "$nodemap_file");
		}
	    }
	}
    }
    #foreach my $n (keys %DestNodeHash){
#	if(!$found{$n}){
#	    $main::dsh_failed{$n}=1;   
#	}
#    }
    foreach my $n (keys %dups){ #get rid of any nodes that map to the same hostname
	    if( scalar (keys %{$dups{$n}}) > 1 ){
		    my @dupNodes = keys %{$dups{$n}};
		    my $dNstring = join ',', @dupNodes;
		    my $hostname = $n;
		    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgHostnameMap', $hostname, $dNstring);
		    foreach my $d (@dupNodes){
			    $main::dsh_failed{$d}=1;
		    }
	    }
    }
    close (NODEMAP);
    if (!@AIX && !@Linux){
        return $::NOK;
    }
    return $::OK;
}

#-------------------------------------------------------------------------#
#                                                                         #
#   start_nfs                                                             #
#                                                                         #
#                                                                         #
#   Return Codes: 0 - All was successful.                                 #
#                 1 - An error occured.                                   #
#-------------------------------------------------------------------------#
sub start_nfs{
    my ($cmd, $output);
    my $rc=0;
    my $exports = "/etc/exports";
    #my $PLTFRM = `uname`; chomp $PLTFRM;      #using $::PLTFRM instead

    # Ensure nfs is running.
    if (! -s $exports){ 
        # nfs will only start if /etc/exports has a size > 0
        open (EXPORTS, ">$exports") || NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_READ_FILE', $exports); 
        print EXPORTS "\n";
        close (EXPORTS);
    }

    # Start nfs if needed.  
    if ($::PLTFRM eq "AIX") {
        my $cmd = "$::MKNFS -B  >/dev/null 2>&1";
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
        my $rc  = system ($cmd) >> 8;
        if ($rc) {
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
        }

    } else {

        # The output of the 'service nfs probe' tells 
        #    what needs to be done to start nfs.  It returns either 
        #    'start', 'restart', 'reload' or nothing.

        $output = `$::SERVICE nfs probe 2>&1`;
        if ($output){
            $cmd = "$::SERVICE nfs $output";
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgRUN_CMD', $cmd);
            $rc  = system ($cmd) >> 8;
            if ($rc) { 	
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCommandFailed', $cmd, $rc);
            }
        }
    }
    return ($rc);
}

#-------------------------------------------------------------------------#
#                                                                         #
#   export_csminstall                                                     #
#                                                                         #
#                                                                         #
#   Return Codes: 0 - All was successful.                                 #
#                 1 - An error occured.                                   #
#-------------------------------------------------------------------------#
sub export_csminstall{
    my ($class, $dir_to_be_exported)= @_;
    my $cmd;
    my $rc=0;
    #my $PLTFRM = `uname`; chomp $PLTFRM;    #using $::PLTFRM instead
    if ( -d $dir_to_be_exported)
    { 
      if ($::PLTFRM eq "AIX") {
        #$cmd = "$::EXPORTFS -i -o ro /csminstall";
        $cmd = "$::EXPORTFS -i -o ro $dir_to_be_exported";
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
        $rc  = system ($cmd) >> 8;
        if ($rc) {
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
        }

      } else {

        #$cmd = "$::EXPORTFS -o ro,no_root_squash \*:/csminstall";
        $cmd = "$::EXPORTFS -o ro,no_root_squash \*:$dir_to_be_exported";
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgRUN_CMD', $cmd);
        $rc  = system ($cmd) >> 8;
        if ($rc) { 
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCommandFailed', $cmd, $rc);
        }
      }
    }
    else
    {
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','W', 'EMsgNo_Dir_Export', $dir_to_be_exported);
        $rc=1;
    }
    return ($rc);
}

#-------------------------------------------------------------------------#
#                                                                         #
#   unexport_csminstall                                                   #
#                                                                         #
#                                                                         #
#   Return Codes: 0 - All was successful.                                 #
#                 1 - An error occured.                                   #
#-------------------------------------------------------------------------#
sub unexport_csminstall{
    my ($class, $dir_to_be_unexported)= @_;
    my $cmd;
    my $rc=0;
    #my $PLTFRM = `uname`; chomp $PLTFRM;      #using $::PLTFRM instead

    if ($::PLTFRM eq "AIX") {
        #$cmd = "$::EXPORTFS -i -u /csminstall";
        $cmd = "$::EXPORTFS -i -u $dir_to_be_unexported";
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
        $rc  = system ($cmd) >> 8;
        if ($rc) { 
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
        }
    } else {

        #$cmd = "$::EXPORTFS -u \*:/csminstall";
        $cmd = "$::EXPORTFS -u \*:$dir_to_be_unexported";
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgRUN_CMD', $cmd);
        $rc  = system ($cmd) >> 8;
        if ($rc) { 
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCommandFailed', $cmd, $rc);
        }
    }
    return ($rc);
}

###########################################################	
# Get the hostname of the management server.
#
# Since the nodes might need to contact the management server through a
# different address than its primary hostname, we cannot just use the output
# of the hostname command here.  Use the getSourceIP2Target command instead.
#
# If we cannot determine the route to the node via getSourceIP2Target
# just return nothing.  This indicates that there is no way to get to the node.
###########################################################

sub get_management_server
{
    my ($class, $node_hostname) = @_;
    my $ipaddr   = "";
    my $hostname = "";

    # Move this to InstallDefs.pm.perl
    $::GETSOURCEIP2TARGET = "/opt/csm/csmbin/getSourceIP2Target";

    my $cmd = "$::GETSOURCEIP2TARGET $node_hostname";
    my @output = NodeUtils->runcmd($cmd,0);
    my $rc = $::RUNCMD_RC;
    if (! $rc) # If no errors
    {
	$ipaddr = $output[0];

	if($ipaddr){
	    $hostname = NodeUtils->tryHost($ipaddr);
	}
    }

    if (wantarray) { return ($hostname, $ipaddr); }
    else { return $hostname; }
}

#################################################
#The following two subroutines are used for locking the tool
#by obtaining a lock at the beginning of execution and holding
#it until the tool completes. Therefore it is guaranteed that two or more
#instances of updatenode will NOT be able to be run.
#
#If process A invokes get_lock and process B invokes it after 
#process A has, process B will fail. 
#
#In the event that the program is aborted, the lock is automatically
#released.
#
#INPUT:
#	get_lock takes the filename you want to use as an argument
#OUTPUT:
#	returns a 2 if it couldn't open the file to lock it
#	returns a 3 if there was an error locking the file
###########################################################

sub get_lock 
{
	my ($class, $file) =@_;
	my $return_code = 0;
	my $program = NodeUtils->programName();
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"V", 'IMsgGet_Lock', $file, $program);
	if(!open(SEM,">$file")) { 
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E1", 'EMsgSEMAPHORE_ERROR', $file, "$? $!");
	}
	if(!flock(SEM,LOCK_EX|LOCK_NB)) 
	{ 
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E1", 'EMsgLOCK_ERROR', $program, $file, "$? $!");
	}
}

#-------------------------------------------------------------------------#
#                                                                         #
#   release lock							  #
#                                                                         #
#       Releases the lock of the file of get lock sub routine             #
#-------------------------------------------------------------------------#
sub release_lock
{
	my ($class, $file) =@_;
	
    close(SEM);
	if(-e "$file"){
		system("/bin/rm -f $file");
	}
}

#-------------------------------------------------------------------------#
#                                                                         #
#   backup_logfile                                                        #
#                                                                         #
#        Backup the current logfile. Move logfile to logfile.1.           #
#        Shift all other logfiles (logfile.[1-3]) up one number.          #
#        The original logfile.4 is removed.                               #
#                                                                         #
#-------------------------------------------------------------------------#
sub backup_logfile
{
    my ($logfile) = @_;
    my ($logfile1) = $logfile . ".1";
    my ($logfile2) = $logfile . ".2";
    my ($logfile3) = $logfile . ".3";
    my ($logfile4) = $logfile . ".4";

    if (-f $logfile) {

        rename ($logfile3, $logfile4) if (-f $logfile3);
        rename ($logfile2, $logfile3) if (-f $logfile2);
        rename ($logfile1, $logfile2) if (-f $logfile1);
        rename ($logfile,  $logfile1);
    }
    return $::OK;
}

#-------------------------------------------------------------------------#
#                                                                         #
#    start_logging                                                        #
#                                                                         #
#           Start logging messages to a logfile. Return the log file      #
#           handle so it can be used to close the file when done logging. #
#                                                                         #
#-------------------------------------------------------------------------#
sub start_logging
{
    my ($class, $logfile, $quiet_flag) = @_;
    my ($cmd, $rc);
	if(!$quiet_flag){
		$quiet_flag =0;
	}
    &backup_logfile($logfile);

    # create the log directory if it's not already there
    if (! -d $::CSMLOG) {
        $cmd = "$::MKDIR -m 644 -p $::CSMLOG";
        $rc = system("$cmd");
        if ($rc >> 8) {
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgNO_CreateDir', $::CSMLOG);
            return($::NOK);
        }
    }

    # open the log file
    unless (open(LOGFILE, ">>$logfile")) {
        # Cannot open file
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgCANT_WRITE_FILE', $logfile);
        return $::NOK;
    }

    $::LOG_FILE_HANDLE = \*LOGFILE;

    # Print the date to the top of the logfile
    my $sdate = `/bin/date`;
    chomp $sdate;
    my $program = NodeUtils->programName();
    if ($quiet_flag !=1)
    {
      NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgWRITING_LOGFILE', $program, $logfile);
      NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"LI", 'IMsgSTART_LOGGING', $sdate);
    }

    return ($::LOG_FILE_HANDLE);
}

#-------------------------------------------------------------------------#
#                                                                         #
#    append_logging                                                       #
#                                                                         #
#         Append logging messages to a logfile. Return the log file       #
#           handle so it can be used to close the file when done logging. #
#                                                                         #
#-------------------------------------------------------------------------#
sub append_logging
{
    my ($class, $logfile, $quiet_flag) = @_;
    my ($cmd, $rc);
if(!$quiet_flag){
		$quiet_flag =0;
	}
    # create the log directory if it's not already there
    if (! -d $::CSMLOG) {
        $cmd = "$::MKDIR -m 644 -p $::CSMLOG";
        $rc = system("$cmd");
        if ($rc >> 8) {
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgNO_CreateDir', $::CSMLOG);
            return($::NOK);
        }
    }

    #  
    #  get log file ready
    #
    if (! -e $logfile) {  #  create the log file if not already there
        # open the log file
        unless (open(LOGFILE, ">$logfile")) {
            # Cannot open file
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgCANT_WRITE_FILE', $logfile);
            return $::NOK;
        }
    } else {   # it's there so just append
        unless (open(LOGFILE, ">>$logfile")) {
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgCANT_WRITE_FILE', $logfile);
            return $::NOK;
        }
    }

    $::LOG_FILE_HANDLE = \*LOGFILE;

    # Print the date to the top of the logfile
    my $sdate = `/bin/date`;
    chomp $sdate;
    my $program = NodeUtils->programName();
    if ($quiet_flag !=1)
    {
       NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgWRITING_LOGFILE', $program, $logfile);
       NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"LI", 'IMsgSTART_LOGGING', $sdate);
    }

    return ($::LOG_FILE_HANDLE);
}

#-------------------------------------------------------------------------#
#                                                                         #
#    stop_logging                                                         #
#                                                                         #
#                  Turn off message logging.  (Expects to have a file     #
#                   handle passed in. )                                   #
#                                                                         #
#-------------------------------------------------------------------------#
sub stop_logging
{
    my ($class, $quiet_flag) = @_;
    if(!$quiet_flag){
    		$quiet_flag =0;
	}
	if (defined($::LOG_FILE_HANDLE)) {
    		# Print the date at the bottom of the logfile
                if ($quiet_flag !=1)
                { 
			my $date = `/bin/date`;
			chomp $date;
       		    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"LI", 'IMsgSTOP_LOGGING', $date);
                }

    		close($::LOG_FILE_HANDLE);
    		undef $::LOG_FILE_HANDLE;
	}
    return $::OK;
}

#-------------------------------------------------------------------------#
#                                                                         #
#    get_OSName                                                           #
#                                                                         #
#                  Returns the Os name i.e. Linux , AIX else undefined    #
#                                                                         #
#                                                                         #
#-------------------------------------------------------------------------#
sub get_OSName
{
    if(NodeUtils->isLinux()==1){
	    return "Linux";
    }
    elsif(NodeUtils->isAIX()==1){
	    return "AIX"
    }
    return undef;
}

#------------------------------------------------------------------------#
#                                                                        #
#    DistributionVersion                                                       #
#                                                                        #
#        Returns the Operating System version that is installed on the   #
#        management server.                                              #
#        For AIX, the version is in the form V.R.M.  For Linux, it is    #
#        in the form reported in the /etc/redhat-release file (V.R).     #
#------------------------------------------------------------------------#
sub get_DistributionVersion
{
    my ($cmd, $output, $rpm, $version, $release, $v1, $v2, $v3, $rest);

    if ($::PLTFRM eq "Linux")
    {
        
        $version = (split ' ', (NodeUtils->distribution()))[1];
        
        #if (-f '/etc/redhat-release')
        #{
        #    my $s = NodeUtils->readFile('/etc/redhat-release');
        #    ($version) = $s =~ /\s(\d+\.\d+)\s/;
        #}
    }
    elsif ($::PLTFRM eq "AIX")
    {
        #$cmd = "$::OSLEVEL";
        $cmd = "$::LSLPP -Lcq bos.rte | /usr/bin/cut -d: -f 3";
        $output = NodeUtils->runcmd($cmd);
        chomp ($output);

        ($v1, $v2, $v3, $rest) = split(/\./, $output);
        $version = join (".", $v1, $v2, $v3);
    }
    else
    {
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E2', 'EMsgINVALID_OSTYPE');
    }

    return (length($version) ? $version : undef);
}

#------------------------------------------------------------------#
#                                                                  #
#    get_CSMVersion                                                #
#                                                                  #
#       Get the version of csm component that is installed on the  #
#       management server.                                         #
#       The version should be in the form "V.R.M" (i.e 1.2.0).     #
#       If showfixlevel is true (1) the version will be in the     #
#       format "V.R.M.F" (i.e. 1.2.0.0)                            #
#                                                                  #
#------------------------------------------------------------------#
sub get_CSMVersion
{
    my ($class, $component, $showfixlevel) = @_;
    my ($cmd, $output, $rpm, $version, $release, $v1, $v2, $v3, $v4, $rest);

    if ($::PLTFRM eq "Linux")
    {
        $cmd = "$::RPMCMD -q $component";
        $output = NodeUtils->runcmd($cmd);
        chomp ($output);
        ($rpm, $version, $release) = split (/-/, $output);

        ($v1, $v2, $v3, $v4, $rest) = split(/\./, $version);
        if ($showfixlevel) { $version = join (".", $v1, $v2, $v3, $v4); }
        else { $version = join (".", $v1, $v2, $v3); }
    }
    elsif ($::PLTFRM eq "AIX")
    {
        $cmd = "$::LSLPP -cLq $component";
        $output = NodeUtils->runcmd($cmd);
        chomp ($output);

        # The third field in the lslpp output is the VRMF
        ($version) = (split(/:/, $output))[2];

        ($v1, $v2, $v3, $v4, $rest) = split(/\./, $version);
        if ($showfixlevel) { $version = join (".", $v1, $v2, $v3, $v4); }
        else { $version = join (".", $v1, $v2, $v3); }
    }
    else
    {
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E2', 'EMsgINVALID_OSTYPE');
    }

    return (length($version) ? $version : undef);
}

#-------------------------------------------------------------------------#
#                                                                         #
#    get_PkgArchitecture                                                  #
#                                                                         #
#        Returns the Os Architure of the OS i.e. i386 for linux           #
#        This only for Linux                                              #
#                                                                         #
#-------------------------------------------------------------------------#
sub get_PkgArchitecture
{
  my ($cmd, $output, $osarch);
 if ($::PLTFRM eq "Linux")
 {
    $cmd = "$::UNAME -m";
    $output = NodeUtils->runcmd($cmd);
    chomp ($output);

    $osarch = $output;

   return (length($osarch) ? $osarch : undef);
 }
 elsif ($::PLTFRM eq "AIX")
 {
     $osarch="";
    return (length($osarch) ? $osarch : undef);
 }
 else
 {
   NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E2', 'EMsgINVALID_OSTYPE');
 }
}

#-------------------------------------------------------------------------#
#                                                                         #
#    get_pkgVersion                                                       #
#                                                                         #
#       Returns the installp image version from Installp command which,   #
#        looks into the .toc file.                                        #
#                                                                         #
#-------------------------------------------------------------------------#
sub get_pkgVersion
{
   my ($class, $pkg) = @_;
   my ($cmd, $output, $version);

   if ($::PLTFRM eq "Linux")
   {
        $version = "";
      return (length($version) ? $version : undef);
   }
   elsif ($::PLTFRM eq "AIX")
   {
      $cmd=qq($::INSTALLPCMD -L -d . | grep $pkg 2>&1);
      $output = NodeUtils->runcmd($cmd);
      chomp ($output);
      (undef,undef,$version,undef)=split(/:/,$output);
      return (length($version) ? $version : undef);
   }
   else
   {
      NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E2', 'EMsgINVALID_OSTYPE');
   }

}

#-------------------------------------------------------------------------#
#                                                                         #
#    get_DistributionName                                                 #
#                                                                         #
#       Get the distribution name for linux its RedHat and return         # 
#       nothing for AIX.						  #
#                                                                         #
#-------------------------------------------------------------------------#
sub get_DistributionName
{
    my ($cmd, $output, $rpm, $distro);

    # IF LINUX
    if ($::PLTFRM eq "Linux")
    {
        $distro = (split ' ', (NodeUtils->distribution()))[0];
        
        #if (-f '/etc/redhat-release')
        #{
            # RedHat is the only Linux distro we support for now.
        #    $distro = "RedHat";
        #}
    }
    elsif ($::PLTFRM eq "AIX")
    {
        $distro = "";
    }
    else
    {
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E2', 'EMsgINVALID_OSTYPE');
    }

    return (length($distro) ? $distro : undef);
}

#-------------------------------------------------------------------------#
#                                                                         #
#    get_uniqu_arry_elemnts                                               #
#                                                                         #
#       Returns the unique elements in an array                           #
#                                                                         #
#-------------------------------------------------------------------------#
sub get_uniqu_arry_elemnts
{
   my ($class, @my_arry) =@_;
   my %my_hsh=();
   my @uniq=();
   my $itm;
   foreach $itm (@my_arry)
   {
     $my_hsh{$itm}++;
   }
   @uniq= keys %my_hsh;
   return @uniq;
}

#-------------------------------------------------------------------------#
#                                                                         #
#    findImagesInPath                                                     #
#                                                                         #
#       Find the images  in the given path                                #
#                                                                         #
#-------------------------------------------------------------------------#
sub findImagesInPath{
    # the $image variable may have a *, so we may find more than one that 
    # matches.

    my ($class, $image, @path) = @_;
    my (@tmpImages,@returnImages,$tmpImage,$realImage);

    foreach my $p (@path){
     	my $filename = "$p/$image";

     	# use the ls command here instead of the -e operator 
     	# so we can handle *s in the names

     	if (`/bin/ls $filename 2> /dev/null`){
       	    # put all of the images we just got into @tmpImages array
       	    chomp(@tmpImages = `/bin/ls $filename`);  

       	    # Go through each tmpImage and see if the base name already
            # exists in the returnImages array.  If it does exist, don't
            # put it in, otherwise, put it in!


       	    foreach $tmpImage (@tmpImages){
	  
	  	# get the base name of the test image. 

	  	my $tstImageBaseName;
		my $defined = 0;
	  	($tstImageBaseName = $tmpImage) =~ s:^.*/::g;
		

	  	foreach $realImage (@returnImages){

	      	    # get the base name of the real image	

	      	    my $realImageBaseName;
	      	    ($realImageBaseName = $realImage) =~ s:^.*/::g;

	       	    if($realImageBaseName eq $tstImageBaseName){
		  	$defined = "true";
	          	last;
	      	    }
	   	}	

	   	unless($defined){
		    push @returnImages, $tmpImage;
	   	}else{

print "\n\nnot copying $tmpImage\n\n";
		}
     	    }
   	} 
    }
    return @returnImages;
}



#-------------------------------------------------------------------------#
#                                                                         #
#    findRpmInPath                                                        #
#                                                                         #
#       Find the rpm package in the given path                            #
#                                                                         #
#-------------------------------------------------------------------------#
sub findRpmInPath
{
   my ($class, $rpmfile, @path) = @_;
   foreach my $p (@path)
   {
     my $filename = "$p/$rpmfile";
     # use the ls command here instead of the -e operator so we can handle *s in the names
     #print "looking for $filename: ", `/bin/ls $filename`, "\n";
     if (`/bin/ls $filename 2> /dev/null`)
     {
       NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgFOUND_RPM', $rpmfile, $p);
       return $p;
     }
   } 
   return undef;
}

#-------------------------------------------------------------------------#
#                                                                         #
#    findRpm                                                              #
#                                                                         #
#       Try to find and return the rpm file that is the required version  #
#                                                                         #
#-------------------------------------------------------------------------#
sub findRpm
{
        my ($class, $pkg, $operator, $version, @my_path) = @_;

        my $os_rpm_path=$my_path[0];
        my $pkg_rpm_path=$my_path[1];
     if ($::PREREQS_ATTR{'OSName'} eq "Linux")
     {
        # Get the current distribution if it was not already obtained.
        my $distro = $::DISTRO;
        if (! $distro)
        {
            $distro = &get_DistributionName." ".&get_DistributionVersion;
            if (!grep(/^$distro$/, @::VALID_DISTROS))
            {
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E5', 'EMsgUNSUPPORTED_DISTRO', $distro, join(', ',@::VALID_DISTROS));
            }
            else
            {
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgDETECTED_DISTRO', $distro);
            }
        }
      }
       # note: The -vr options on the ls command gives us the entries in
        #       reverse version order to get the latest version first.
        my @rpms;
        my @rpms1;
        my @rpms2;

        # Look for the rpm in packages
        if (NodeUtils->isLinux)
        {
           @rpms1 = NodeUtils->runcmd("$::LS -vr $pkg_rpm_path/$pkg-*", -1);
        }
        else
        {
           @rpms1 = NodeUtils->runcmd("$::LS  $pkg_rpm_path/$pkg-*", -1);
        }
        my $rc1 = $::RUNCMD_RC;
        if (! $rc1) { @rpms = (@rpms, @rpms1); } # Only add if no errors

        # Look for the rpm in  RPMS
        #my $TFTPBOOT_DISTRORPM = "$::TFTPBOOT/$::DISTRO_SHORTNAME{$distro}/$::DISTRO_RPMDIR{$distro}";
        if (NodeUtils->isLinux)
        {
           @rpms2 = NodeUtils->runcmd("$::LS -vr $os_rpm_path/$pkg-*", -1);
        }
        else
        {
           @rpms2 = NodeUtils->runcmd("$::LS $os_rpm_path/$pkg-*", -1);
        }
        my $rc2 = $::RUNCMD_RC;
        if (! $rc2) { @rpms = (@rpms, @rpms2); } # Only add if no errors

        if (($rc1 && $rc2) || !scalar(@rpms)) { return undef; }

        foreach my $r (@rpms)
          {
                #my ($ipkg, $iversion, $irelease) = split('-', $r);
                my ($fn) = $r =~ m|^.*?([^/]+)$|;      # get rid of the directory, if any
                #print "fn=$fn\n";
                my ($ipkg, $iversion, $irelease) = $fn =~ /^(.+)-(\d.*)-(.*)$/;
                #print "ipkg=$ipkg, iversion=$iversion, irelease=$irelease\n";
                # Note: irelease has .i386.rpm on the end of it
                if ($ipkg ne $pkg || !length($iversion)) { next; }
                if (!length($operator) || !length($version) || InstallUtils->testVersion($iversion, $operator, $version))
                  {
                        $operator=" " if (!$operator);
                        $version=" " if (!$version);

                        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgFOUND', $r, $operator, $version);
                        return $r;
                  }
          }
        return undef;   # didn't find a suitable rpm

}

#-------------------------------------------------------------------------#
#                                                                         #
#   testVersion                                                           #
#                                                                         #
#       Compare version1 and version2 according to the operator and       #
#       return true or false. These expressions come from the Require     #
#       statement of our rpms.						  #
#                                                                         #
#-------------------------------------------------------------------------#
sub testVersion
  {
	my ($class, $version1, $operator, $version2, $release1, $release2) = @_;

	# Note: perl seems to be able to handle really big integers, but if we run into a limit, we
	#       can use the Math::BigInt pkg

	my @a1 = split(/\./, $version1);
	my @a2 = split(/\./, $version2);
	my $len = (scalar(@a1)>scalar(@a2) ? scalar(@a1) : scalar(@a2));
	$#a1 = $len -1;      # make the arrays the same length before appending release
	$#a2 = $len -1;
	push @a1, split(/\./, $release1);
	push @a2, split(/\./, $release2);
	$len = (scalar(@a1)>scalar(@a2) ? scalar(@a1) : scalar(@a2));
	my $num1 = '';
	my $num2 = '';

	for (my $i=0; $i<$len; $i++)
	  {
		my ($d1) = $a1[$i] =~ /^(\d*)/;     # remove any non-numbers on the end
		my ($d2) = $a2[$i] =~ /^(\d*)/;
		
		my $diff = length($d1) - length($d2);
		if ($diff > 0)     # pad d2
		  {
			$num1 .= $d1;
			$num2 .= ('0' x $diff) . $d2;
		  }
		elsif ($diff < 0)     # pad d1
		  {
			$num1 .= ('0' x abs($diff)) . $d1;
			$num2 .= $d2;
		  }
		else        # they are the same length
		  {
			$num1 .= $d1;
			$num2 .= $d2;
		  }
	  }

	# Remove the leading 0s or perl will interpret the numbers as octal
	$num1 =~ s/^0+//;
	$num2 =~ s/^0+//;

	if (length($release1)) { $release1 = "-$release1"; }   # this is just for error msgs
	if (length($release2)) { $release2 = "-$release2"; }   # this is just for error msgs
	
	if ($operator eq '=') { $operator = '=='; }
	my $bool = eval "$num1 $operator $num2";
	if (length($@))
	  {
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','W', 'EMsgVERSION_COMPARE_PROBLEM', "$version1$release1 $operator $version2$release2", $@);
		return undef;
	  }
	if($ENV{LANG} =~ /^(C|POSIX|en_US)/){  #this is a work around for NLS problems
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCOMPARING_VERSION', "$version1$release1 $operator $version2$release2", ($bool?'true':'false'));
	}
	return $bool;
  }

#-------------------------------------------------------------------------#
#                                                                         #
#    installRequiredRpms                                                  #
#                                                                         #
#     Install any rpms that this rpm is dependent on if they              #
#     are not already on the system at the required level.                #
#     (The rpm passed in is the whole rpm file name.)                     #
#                                                                         #
#-------------------------------------------------------------------------#
sub installRequiredRpms
{
        my ($class, $rpm, @my_path) = @_;
        my @reqs = InstallUtils->getRequiredRpms($rpm, @my_path);
        if (scalar(@reqs))        # need to install at least 1 rpm
          {
                my $reqlist = join(' ', @reqs);
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V','IMsgINSTALLING_RPM_REQS', $rpm, $reqlist);
                my ($cmd) = "$::RPMCMD -U $reqlist";
                $cmd = "$cmd --force" if $::FORCE_INSTALL;
                NodeUtils->runcmd("$cmd");
          }

}

#-------------------------------------------------------------------------#
#                                                                         #
#    getRequiredRpms                                                      #
#                                                                         #
#       Return the names of any rpms that this rpm is dependent on if     #
#       they are not already on the system at the required level.         #
#       The rpm passed in as the 1st arg is the whole rpm file name.      # 
#       The rest of the args are the list of reqs we have already found   #
#       in previous, recursive invokations of the function.               #
#                                                                         #
#-------------------------------------------------------------------------#
sub getRequiredRpms
{
        my ($class, $rpm, @my_path) = @_;
        my @reqrpms;
        my @reqs = InstallUtils->getReqsNotSatisfied($rpm);
        #print "num=" . scalar(@reqs) . ' ,reqs=' . join(':', @reqs) . ".\n";
        foreach my $r (@reqs)
          {
                my ($pkg, $operator, $version) = split(' ', $r);

                my $newrpm = InstallUtils->findRpm($pkg, $operator, $version, @my_path);
                if (defined($newrpm))        # found an appropriate one to install
                  {
                        # make sure it is not already on the list
                        if (grep(/^$newrpm$/, @_)) { next; }   # already on the list

                        push @reqrpms, $newrpm;

                        push @reqrpms, InstallUtils->getRequiredRpms($newrpm, @reqrpms, @my_path);
                  }
                else { NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E11', 'EMsgCANT_FIND_RPM', $pkg, $operator, $version, $rpm); }
          }
        return @reqrpms;
}

#-------------------------------------------------------------------------#
#                                                                         #
#    getReqsNotSatisfied                                                  #
#                                                                         #
#       Return the rpms (and versions) that the given rpm requires        #
#       that are not already on the system                                #
#                                                                         #
#-------------------------------------------------------------------------#
sub getReqsNotSatisfied
{
        my ($class, $rpm) = @_;
        my @requires;
        my @output = NodeUtils->runcmd("$::RPMCMD -i --test $rpm", -1);
        if ($::RUNCMD_RC == 0) { return @requires; }    # all dependencies were satisfied
        if ($::RUNCMD_RC > 1) { return @requires; }
        my $firstline = shift @output;
        if ($firstline=~/is already installed/) { return @requires; }
        if (!($firstline=~/failed dependencies/)) { return @requires; }

        foreach my $o (@output)
          {
                #print "o=$o\n";
                my ($pkg, $operator, $version) = InstallUtils->parseRequire($o);
                #print "pkg=$pkg, operator=$operator, version=$version\n";
                if (!defined($pkg)) { next; }   # was not an rpm require

                # Hack for glibc version because I can not find a way to have the require
                # statement in the rpm be different for different distribution levels.
                if ($pkg eq 'glibc' && NodeUtils->distribution()=~/^RedHat 7/i)
                  { $o = "$pkg $operator 2.2"; }

                my $req = "$pkg $operator $version";
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgREQUIRE', $rpm, $req);
                push(@requires, $req);
          }
        return @requires;
}

#-------------------------------------------------------------------------#
#                                                                         #
#    parseRequire                                                         #
#                                                                         #
#       Parse a single require line, distinguishing between rpms and      #
#       individual libraries of executables.                              #
#                                                                         #
#-------------------------------------------------------------------------#
sub parseRequire
{
        my ($class, $line) = @_;
        $line =~ s/ is needed by .*$//;
        my ($pkg, $operator, $version, $therest) = split(' ', $line);
        if ($pkg=~/^rpmlib/ || $pkg=~m|/| || $pkg=~/\.so/) { return undef; }
        else { return ($pkg, $operator, $version); }
}


#-------------------------------------------------------------------------#
#									  #
#      inc_ip								  #
#           Increment an IP address by 1, carrying to the previous octet  #
#           as required. This code originally appeared in the LUI 	  #
#           product (luiapi.pm)						  #
#                                                                         #
#-------------------------------------------------------------------------#
sub inc_ip
{
    my $ip=$_[1];
    my ($a,$b,$c,$d);
    ($a,$b,$c,$d)=split(/\./,$ip);
    $d++;
    if ($d > 255) {
        $d = $d - 256;
        $c++;
        if ($c > 255) {
            $c = $c - 256;
            $b++;
            if ($b > 255) {
                $b = $b - 256;
                $a++;
                if ($a > 255) {
                    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgIPADDR_ERROR', "$a.$b.$c.$d");
                    #printf "Error - ip address out of valid range.\n";
                    return 1;
                }
            }
        }
    }
    return $a."\.".$b."\.".$c."\.".$d;

}



#-------------------------------------------------------------------------#
#									  #
#      genHex								  #
#      generates a hex representation of an ip address:                   #
#      for example, 9.114.133.193 --> 097285C1                            #
#      Takes in a list of hosts                                           #
#      and returns a hex where the key is the hostname and                #
#                             the value is the IP in hex form.            #
#-------------------------------------------------------------------------#

sub genHex{
    shift;
    my $host;
    my $uno;
    my $dos;
    my $tres;
    my $cuatro;
    my $hex;
    my %hex;
    foreach $host (@_){
        my $ip =  NodeUtils->getHost($host);            # Get ip address
        ($uno, $dos, $tres, $cuatro) = split(/\./, $ip);# Split by the .
        $hex = sprintf("%02X", $uno);                   # put formatted into hex
        $hex .= sprintf("%02X", $dos);
        $hex .= sprintf("%02X", $tres);
        $hex .= sprintf("%02X", $cuatro); 
        $hex{$host} = $hex;
    }
    return %hex;
}

#-------------------------------------------------------------------------#
#									  #
#      setup_atftp  						          #
#									  #
#	   Create the /etc/xinetd.d/tftp file to work with atftp.         #
#									  #
#									  #
#-------------------------------------------------------------------------#
sub setup_atftp
{
    my $xinetd         = "/etc/xinetd.d";
    my $xinetdfile     = $xinetd . "/tftp";
    my $xinetdbackup   = "/tmp/tftp.bak";
    my $atftp_conf     = "/opt/csm/install/atftp_xinetd.conf";
    my $errors         = 0;
    my $DIFF           = "/usr/bin/diff";
    my ($cmd, $output, $rc);

    unless (-d $xinetd ){
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgXINETD_MISSING', "$xinetd");
        $errors++;
    }

    # Only copy the file if it is different
    $cmd = "$DIFF $atftp_conf $xinetdfile 2>&1";
    $output = NodeUtils->runcmd($cmd,-1);
    $rc = $::RUNCMD_RC;
    if ($rc)
    {
	# Back up any existing file
	if (-f $xinetdfile)
	{
	    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"I", 'IMsgCHANGING_TFTP_XINETD_FILE');
	    $cmd = "$::COPY -f $xinetdfile $xinetdbackup";
	    $output = NodeUtils->runcmd($cmd,0);
	    $rc = $::RUNCMD_RC;
	    if ($rc)
	    {
		#print $output;
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','I','IMsgShow_Output',$output);
		$errors++;
	    }
	}

	# Now, copy our version of the file
	$cmd = "$::COPY -f $atftp_conf $xinetdfile";
	$output = NodeUtils->runcmd($cmd,0);
	$rc = $::RUNCMD_RC;
	if ($rc)
	{
	    #print $output;
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','I','IMsgShow_Output',$output);
	    $errors++;
	}

	# Restart xinetd to pick up the changes
	`$::SERVICE xinetd start`;
	`$::SERVICE xinetd reload`;
    }

    return $errors;
}


#-------------------------------------------------------------------------#
#									  #
#      verify_tftp  						          #
#									  #
#	   check that tftp server is running in /etc/xinetd.  Also check  #
#          that the client program is found in $::ATFTP.                  #
#									  #
#									  #
#-------------------------------------------------------------------------#

sub verify_tftp 
{
    my $errors = 0;
    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCheckingAtftp');
    if(-e $::ATFTP){
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgfoundAtftp', $::ATFTP);
        my $version = `$::ATFTP -V`;              # see what version

        chomp(my @output = `$::LSOF | $::GREP tftp | $::GREP xinetd`);
        if(@output){
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgAtftpRunning');
        }
        else{
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgAtftpNotUp');
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgAtftpTryStart');
            `$::SERVICE xinetd start`;
            `$::SERVICE xinetd reload`;
            chomp(my @output = `$::LSOF | $::GREP tftp | $::GREP xinetd`);
            if(@output){
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgAtftpNowRunning');
            }
            else{
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgAtftpCantUp');
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgAtftpXinetdExistquest');
                if(-f "/etc/xinetd.d/tftp"){
                    NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgAtftpXinetdExist');
                }
                else{ NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgAtftpUrProblem'); }
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgAtftpProblemHand');
                $errors++;
            }
        }

    }
    else{
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','I', 'IMsgAtftpNotFound',$::ATFTP);
        $errors++;
    }
    return $errors;
}

#-------------------------------------------------------------------------#
#									  #
#      verify_dhcp  						          #
#									  #
#          check that dhcp is up and running.  Right now we don't check   #
#          the /etc/dhcpd.conf file, since that should've been checked    #
#          when made.  This function is just for extreme paranoya.        #
#									  #
#-------------------------------------------------------------------------#

sub verify_dhcp
{
        my $errors = 0;
        my @output;
        my $cmd;
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgDhcpCheck');
        unless(-f "/var/lib/dhcp/dhcpd.leases"){
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','I', 'IMsgDhcpMissing');
            unless(-d "var/lib/dhcp"){
                NodeUtils->runcmd("$::MKDIR -p /var/lib/dhcp");
            }
            NodeUtils->runcmd("$::TOUCH /var/lib/dhcp/dhcpd.leases");
        }
        $cmd = "$::SERVICE dhcpd status";
        chomp(@output = `$cmd`);
        my @line = split(" ",@output);
        if( grep /running.../, @output){
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgDhcpOutput', @output);
        }
        else{
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgDhcpOutput', @output);
            NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgDhcpProblemRestart', @output);
            `$::SERVICE dhcpd start`;
            $cmd = "$::SERVICE dhcpd status";
            chomp(@output = `$cmd`);
            if( grep /running.../, @output){
                #print "dhcp: @output\n" if $::VERBOSE;
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgDhcpOutput', @output);
            }
            else{
                NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','I', 'IMsgDhcpNotWorking');
                $errors++;
            }
        }
        return $errors;
}

#-------------------------------------------------------------------------#
#									  #
#      testvalidconsole					                  #
#									  #
#            checks for the valid Console Method attributes. This         #
#            subroutine  compares the given Console method against the    #
#            list in the /opt/csm/bin/*_console Console methods.          #
#									  #
#-------------------------------------------------------------------------#
sub testvalidconsole
{
  my ($class, $my_console) = @_;
  my @temp_console=();
  @::valid_console=();
  my $console="";
  my $tmp_console="";
  @temp_console=NodeUtils->runcmd("$::LS \/opt\/csm\/bin\/\*_console", -1);
  if ($::RUNCMD_RC != 0)
  {
    return 2; 
  }
  foreach $tmp_console (@temp_console)
  {
     (undef, undef, undef,undef, $console)=split("/",$tmp_console);
     ($console, undef)=split("_",$console);
     push(@::valid_console, $console);
  }
  if (!grep(/^$my_console$/, @::valid_console))
  {
    return 1;
  }
  else
  {
    return 0;
  }
}


#-------------------------------------------------------------------------#
#									  #
#      testvalidpowermethod                                               #
#									  #
#            checks for the valid Power Method attributes. This           #
#            subroutine  compares the given Power method against the one  #
#            in the /opt/csm/lib/lib+<given Power Method>+.so. If this    #
#            Method is not found then it checks against the               #
#            /opt/csm/bin/*_power Power Methods                           #
#									  #
#-------------------------------------------------------------------------#
sub testvalidpowermethod
{
  my ($class, $my_powermethod) = @_;
  my @temp_powermethod=();
  @::valid_powermethod=();
  my $powermethod="";
  my $tmp_powermethod="";
  #my $my_temp_powermethod="lib".$my_powermethod.".so";
  my $my_temp_powermethod="lib".$my_powermethod;
  my $exit_code=0;
  @temp_powermethod=();
  @temp_powermethod=NodeUtils->runcmd("$::LS \/opt\/csm\/lib\/\*_power\.so", -1);
  if ($::RUNCMD_RC != 0)
  {
    $exit_code=2;
  }
  else
  {
    foreach $tmp_powermethod (@temp_powermethod)
    {
      (undef, undef, undef,undef, $powermethod)=split("/",$tmp_powermethod);
      ($powermethod, undef)=split("_",$powermethod);
      push(@::valid_powermethod, $powermethod);
    }
    if (!grep(/^$my_temp_powermethod$/, @::valid_powermethod))
    {
      $exit_code=1; 
    }
    else
    {
      $exit_code=0; 
    }
  }
  if ( ($exit_code ==1) || ($exit_code ==2) )
  {
    @temp_powermethod=();
    $my_temp_powermethod=$my_powermethod;
    @temp_powermethod=NodeUtils->runcmd("$::LS \/opt\/csm\/bin\/\*_power", -1);
    if ($::RUNCMD_RC != 0)
    {
      #$exit_code=2;
    }
    else
    {
      foreach $my_temp_powermethod (@temp_powermethod)
      {
        (undef, undef, undef,undef, $powermethod)=split("/",$my_temp_powermethod);
        ($powermethod, undef)=split("_",$powermethod);
        push(@::valid_powermethod, $powermethod);
      }

      if (!grep(/^$my_temp_powermethod$/, @::valid_powermethod))
      {
        $exit_code=1; 
      }
      else
      {
        $exit_code=0; 
      }
    } 
   }
   return $exit_code;
# end new code
}

#-------------------------------------------------------------------------#
#									  #
#      chkethup                                                           #
#          Ensure that the eth0 is up and running                         #
#									  #
#-------------------------------------------------------------------------#
sub chkethup
{
    my $cmd = "$::NETSTAT -rn | $::GREP eth0";
    unless(`$cmd`)
    {
        NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E1','EMsgEth0NotUp');
    }
  return;
}

#-------------------------------------------------------------------------#
#									  #
#      cpSSHFiles                                                         #
#
#	returns: 1 if it failed					
#		or locations of authorized keys in /csminstall
#-------------------------------------------------------------------------#
sub cpSSHFiles {
	my ($cmd, $rc);
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCopySSHKeys');
	my $SSHdir = "/csminstall/csm/config/.ssh";
	if( !(-d "/csminstall/csm/config")){
		$cmd = "$::MKDIR /csminstall/csm/config";
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
		`$cmd 2>&1`;
		$rc  = $? >> 8;
		if ($rc) {
			NodeUtils->message('E', 'EMsgNO_CreateDir', "/csminstall/csm/config");
		}
	}
	if( !(-d "$SSHdir")){
        	$cmd = "$::MKDIR $SSHdir";
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
		`$cmd 2>&1`;
		$rc  = $? >> 8;
		if ($rc) {
			NodeUtils->message('E', 'EMsgNO_CreateDir', $SSHdir);
        	}
	}
	my $home = InstallUtils->getRootHomeDir();
	my $authorized_keys = "$SSHdir/authorized_keys";
        my $authorized_keys2 = "$SSHdir/authorized_keys2";
        if(!(-e "$home/.ssh/identity.pub") || !(-e "$home/.ssh/id_rsa.pub") || !(-e "$home/.ssh/id_dsa.pub") ){
		return 1;
	}
	$cmd = "$::COPY $home/.ssh/identity.pub $authorized_keys";
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
	`$cmd 2>&1`;
	$rc  = $? >> 8;
	if ($rc) {
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
        }
        
        $cmd = "$::COPY $home/.ssh/id_rsa.pub $authorized_keys2";
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
	`$cmd 2>&1`;
	$rc  = $? >> 8;
	if ($rc) {
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
        }
        
	$cmd = "$::CAT $home/.ssh/id_dsa.pub >> $authorized_keys2";
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
	`$cmd 2>&1`;
	$rc  = $? >> 8;
	if ($rc) {
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
        }
        if(!(-e "$authorized_keys") || !(-e "$authorized_keys2")){
		return 1;
	}
        return ($authorized_keys, $authorized_keys2);
}

#-------------------------------------------------------------------------#
#									  #
#      installSSHFiles                                                    #
#
#	input: 
#		$filename_of_authorized_keys
#		$filename_of_authorized_keys2
#-------------------------------------------------------------------------#
sub installSSHFiles{
	my ($class, $authorized_keys, $authorized_keys2) = @_;
	my ($cmd, $rc);
	my $home = InstallUtils->getRootHomeDir();
	my $dest_dir = "$home/.ssh";
	my $dest_authorized_keys = "$dest_dir/authorized_keys";
	my $dest_authorized_keys2 = "$dest_dir/authorized_keys2";
	
	umask(0077);
	if (! -d "$dest_dir" ) {
	    # create a local directory
	    $cmd = "$::MKDIR -p $dest_dir";
	    $rc = system("$cmd");
	    if ($rc >> 8){
		# Could not create $::CSMINSTDIR directory.
		NodeUtils->message('E', 'EMsgNO_CreateDir', $dest_dir);
		return $::NOK;
	    }
	    else{
		    chmod 0700, $dest_dir;
	    }
	}

	umask(0133);
	if( -e "$dest_authorized_keys"){
		$cmd = "$::CAT $authorized_keys >> $dest_authorized_keys";
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
		`$cmd 2>&1`;
		$rc  = $? >> 8;
		if ($rc) {
			NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
        	}
	}
	else{
		#just copy the keys
		my $cmd = "$::COPY $authorized_keys $dest_authorized_keys";
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
		`$cmd 2>&1`;
		$rc  = $? >> 8;
		if ($rc) {
			NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
        	}
		chmod 0644, $dest_authorized_keys;
	}
	if( -e "$dest_authorized_keys2"){
		$cmd = "$::CAT $authorized_keys2 >> $dest_authorized_keys2";
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
		`$cmd 2>&1`;
		$rc  = $? >> 8;
		if ($rc) {
			NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
		}
	}
	else{
		#just copy the keys
		my $cmd = "$::COPY $authorized_keys2 $dest_authorized_keys2";
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
		`$cmd 2>&1`;
		$rc  = $? >> 8;
		if ($rc) {
			NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
        	}
		chmod 0644, $dest_authorized_keys2;	
	}
}

#-------------------------------------------------------------------------#
#									  #
#      checkHOMEforSSHperm                                                #
#	
#	returns: 1 if $HOME has the wrong permissions -- SSH should not be setup
#		 0 if the permissions are correct
#-------------------------------------------------------------------------#
sub checkHOMEforSSHperm{
	my $home = InstallUtils->getRootHomeDir();
	my $mode = (stat($home))[2];
	my $real_mode = sprintf "%04o", $mode & 07777;
	my @perm = split '', $real_mode;
	if( ($perm[2] == 2) || ($perm[2] == 6) || ($perm[2] == 7) || ($perm[3] == 2) || ($perm[3] == 6) || ($perm[3] == 7) ){
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgWrongHomePermissions', $home);
		return 1;
	}
	else{
		return 0;
	}
}

#-------------------------------------------------------------------------#
#									  #
#      getRemoteShell                                                     #
#
#	returns: 1 if it failed
#		 "path_of_remote_shell" otherwise
#-------------------------------------------------------------------------#
sub getRemoteShell {
	# set local scope because we only want classes on the mgmt svr
	$ENV{'CT_MANAGEMENT_SCOPE'} = 1;
	# todo: remove when lsrsrc-api converts to above
	$ENV{'CT_SESSION_SCOPE'} = 1;
	if(NodeUtils->isMgmtSvr()){ #only defined on managment server
		if(!$ENV{'DSH_REMOTE_CMD'}){
			my $outref = NodeUtils->runrmccmd('lsrsrc-api', "-i", "-s IBM.DmsCtrl::::RemoteShell");
			return $$outref[0];
		}
		else{
			return $ENV{'DSH_REMOTE_CMD'};
		}
	}
	return 1;
}

#-------------------------------------------------------------------------#
#									  #
#      getSetupRemoteShell                                                     #
#
#	returns: value of SetupRemoteShell in CSM database, or null
#-------------------------------------------------------------------------#
sub getSetupRemoteShell {
	# set local scope because we only want classes on the mgmt svr
	$ENV{'CT_MANAGEMENT_SCOPE'} = 1;
	# todo: remove when lsrsrc-api converts to above
	$ENV{'CT_SESSION_SCOPE'} = 1;
	if(NodeUtils->isMgmtSvr()){ #only defined on managment server
			my $outref = NodeUtils->runrmccmd('lsrsrc-api', "-i", "-s IBM.DmsCtrl::::SetupRemoteShell");
			return $$outref[0];
	}
}

#-------------------------------------------------------------------------#
#									  #
#      setupLinuxRSH	                                                  #
#
#	input:
#		$full_hostname_of_management_server			  #
#-------------------------------------------------------------------------#
sub setupLinuxRSH{
	my ($class, $ms) = @_;
	umask(0177);
	my $home = InstallUtils->getRootHomeDir();
	my ($cmd, $rc);
	$cmd = "/bin/sed -e \"s/disable.*=.*yes/disable = no/\" /etc/xinetd.d/rsh";
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
	my @rsh = `$cmd 2>&1`;
	$rc  = $? >> 8;
	if ($rc) {
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
	}
	open (RSH, ">/etc/xinetd.d/rsh") or NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall',"E", 'EMsgCANT_WRITE_FILE', "/etc/xinetd.d/rsh");
	print RSH @rsh;
	close RSH;
	chmod 0600, "/etc/xinetd.d/rsh";
	$cmd = "/sbin/service xinetd restart";
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
	`$cmd 2>&1`;
	$rc  = $? >> 8;
	if ($rc) {
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
	}
	if(-e "/etc/securetty"){
		$cmd = "/bin/grep rsh /etc/securetty";
		`$cmd 2>&1`;
		$rc  = $? >> 8;
		if($rc){
			`/bin/echo rsh >> /etc/securetty`;
		}
	}
	$cmd = "/bin/echo $ms root >> $home/.rhosts";
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
	`$cmd 2>&1`;
	$rc  = $? >> 8;
	if ($rc) {
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
	}
	chmod 0600, "$home/.rhosts";
}

#-------------------------------------------------------------------------#
#									  #
#      setupAIXRSH	                                                  #
#
#	input:
#		$full_hostname_of_management_server			  #
#									  #
#-------------------------------------------------------------------------#
sub setupAIXRSH{
	my ($class, $ms) = @_;
	umask(0177);
	my $home = InstallUtils->getRootHomeDir();
	my ($cmd, $rc);
	$cmd = "/bin/echo $ms root >> $home/.rhost";
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cmd);
	`$cmd 2>&1`;
	$rc  = $? >> 8;
	if ($rc) {
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cmd, $rc);
	}
	chmod 0600, "$home/.rhosts";
}

#-------------------------------------------------------------------------#
#									  #
#      supportedSSHversion                                                #
#                							  #
#	 requires $::REMOTE_SHELL to be set				  #
#									  #
#	 returns 1 if the version of SSH is supported by install	  #
#		 0 if not supported -- install should not setup ssh	  #
#									  #
#-------------------------------------------------------------------------#
sub supportedSSHversion{
	#check if its a supported version of OpenSSH
	my $ssh_check = "$::REMOTE_SHELL -V";
	my @ssh_res = `$ssh_check 2>&1`;
	my $ssh_rc = $? >>8;
	if(!$ssh_rc){ #successful
		if($ssh_res[0] =~ m/OpenSSH.(\d)\.\d/){ #we can set this up
			if($1 >= 2){
				return 1;
			}
		}
		else{
			return 0;
		}
	}
}

#-------------------------------------------------------------------------#
#									  #
#      setupRemoteShellforFullInstall                                     #
#           Determine which shell to set up (rsh or ssh) and do           #
#           everything that is required on the management server.  Also,  #
#           set the $::SETUP_REMOTE_SHELL global varible to 1 if the      #
#           remote shell should be set up on the nodes.  This used by     #
#           make_config_file to set the SetupRemoteShell attribute.       #
#									  #
#           This subroutine is called by installnode and csmsetupnim.	  #
#									  #
#        sets these global variables:                                     #
#           $::REMOTE_SHELL       : The remote shell to set up.           #
#           $::SETUP_REMOTE_SHELL : If 1, setup remote shell on nodes     #
#                                   If 1, do not set up remote shell      #
#									  #
#	 returns 1 if there was a problem                                 #
#		 0 no problems                                            #
#									  #
#-------------------------------------------------------------------------#
sub setupRemoteShellforFullInstall
{
    my $rc = 0;

    my ($class, $ref_nodes) = @_;
    my @node_hostnames = @$ref_nodes;

    # SETUP_REMOTE_SHELL controls whether or not to set up the remote shell 
    # on the nodes.  It is used when creating the config_file.
    #     0=do not set up remote shell
    #     1=set up whichever remote shell is listed in $::REMOTE_SHELL
    $::SETUP_REMOTE_SHELL = InstallUtils->getSetupRemoteShell();

    # If the SetupRemoteShell attribute is set to 0 (or null) in DMSCTRL,
    # don't do any remote shell setup.
    if (! $::SETUP_REMOTE_SHELL)
    {
	return $rc;
    }

    $::REMOTE_SHELL = InstallUtils->getRemoteShell();

    # If the REMOTE_SHELL is set to 1, there was an error getting it from
    # the dmsctrl.  Don't bother setting up the remote shell.
    if ($::REMOTE_SHELL == 1)
    {
	return $::NOK;
    }

    (my $cmd_basename = $::REMOTE_SHELL) =~ s:^.*/::;
    if ($cmd_basename eq "rsh")
    {
	$::SETUP_REMOTE_SHELL = 1;
    }
    else # Attempt to set up ssh
    {
	if (InstallUtils->supportedSSHversion())
	{
	    # Check permissions on home directory
	    $rc = InstallUtils->checkHOMEforSSHperm();
	    if ($rc)
	    {
		$::SETUP_REMOTE_SHELL = 0;
		return $rc;
	    }
  
	    # Setup ssh keys
	    NodeUtils->runcmd("/opt/csm/csmbin/remoteshell.expect -k", 0);

	    # Copy ssh keys to /csminstall
	    $rc = InstallUtils->cpSSHFiles();
	    if ($rc == 1)
	    {
		$::SETUP_REMOTE_SHELL = 0;
		return $rc;
	    }

	    # Clean up the Known Hosts file for each node so that when the 
	    # nodes are reinstalled, the hosts files will be regenerated.
	    $rc = InstallUtils->rmNodesKnownHosts(\@node_hostnames);
	    

	    # ssh was set up properly, so set it up on the nodes.
	    $::SETUP_REMOTE_SHELL = 1;

	    # Add RemoteShell to config_file
	}
    }

    return $rc;
}

#-------------------------------------------------------------------------#
#									  #
#      rmNodesKnownHosts                                                  #
#                							  #
#       input:								  #
#		\@node_hostnames					  #
#									  #
#	returns 	1 if failure					  #	
# 			0 if success					  #
#-------------------------------------------------------------------------#
sub rmNodesKnownHosts{
	my ($class, $ref_nodes) = @_;
	my @node_hostnames = @$ref_nodes;
	my $home = InstallUtils->getRootHomeDir();
	my @all_names;
	my $return =0;
	foreach my $n (@node_hostnames){
		my @parts = split /\./, $n;
		my $name = shift @parts;
		push @all_names, $name;
		foreach my $ext (@parts){
			$name = "$name.$ext";
			push @all_names, $name;
		}
	}
	#create the sed command
	my $sed = "/bin/sed -e ";
	$sed .= "\"";
	foreach my $n (@all_names){
		$sed .= "/^$n\[,| ]/d; "
	}
	chop $sed; #get rid of last space
	$sed .= "\"";
	my $sed2 = $sed;
	my $file = "$home/.ssh/$$";
	while(-e $file){
		$file = InstallUtils->CreateRandomName($file);
	}
	if(-e "$home/.ssh/known_hosts"){
		$sed .= " $home/.ssh/known_hosts";
		$sed .= " > $file";
		my $printsed = $sed;
		$printsed =~ s/"//g; #"
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $printsed);
		`$sed 2>&1`;
		my $rc  = $? >> 8;
		if ($rc) {
			$return =1;
			NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $printsed, $rc);
		}
		my $cp = "$::COPY $file $home/.ssh/known_hosts";
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cp);
		`$cp 2>&1`;
		$rc  = $? >> 8;
		if ($rc) {
			$return =1;
			NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cp, $rc);
		}
		system("$::RM -f $file");
	}
	if(-e "$home/.ssh/known_hosts2"){
		$sed2 .= " $home/.ssh/known_hosts2";
		$sed2 .= " > $file";
		my $printsed2 = $sed2;
		$printsed2 =~ s/"//g; #"
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $printsed2);
		`$sed2 2>&1`;
		my $rc  = $? >> 8;
		if ($rc) {
			$return =1;
			NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $printsed2, $rc);
		}
		my $cp = "$::COPY $file $home/.ssh/known_hosts2";
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $cp);
		`$cp 2>&1`;
		$rc  = $? >> 8;
		if ($rc) {
			$return =1;
			NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $cp, $rc);
		}
		system("$::RM -f $file");	
	}
	return $return;
}

#-------------------------------------------------------------------------#
#									  #
#      getRootHomeDir                                                     #
#                							  #
#	returns 	$path_of_root_home_dir				  #	
# 									  #
#-------------------------------------------------------------------------#
sub getRootHomeDir{
	my @root = split ':', (`/bin/grep ^root /etc/passwd 2>&1`);
	my $home = $root[5];
	return $home;
}
#-------------------------------------------------------------------------#
#                                                                         #
#     getRecentVersionRpm                                                 # 
#                                                                         #
#       returns         $recent rpm                                       #
#                                                                         #
#-------------------------------------------------------------------------#
sub getRecentVersionRpm
{
  my ($class, $path, $rpm_name) = @_;

  my @localversions=();
  my %packver;
  my @output=`ls  -1 $path/$rpm_name`;
  chomp @output;
  foreach my $rpm1 (@output)
  {
    my $VERSION=`rpm -qp --qf '%{Version}' $rpm1`;
    chomp $VERSION;
    $packver{"$VERSION"}=$rpm1;
    push (@localversions, $VERSION);
  }
  my $versionout="";
  my $i=0;
  foreach my $my_version (@localversions)
  {
    if ($i==0) {$versionout=$my_version; }
    if (InstallUtils->testVersion($my_version,">",$versionout))
    {
      $versionout=$my_version;
    }
    $i++;
  }

  return ($packver{$versionout});
}

#-------------------------------------------------------------------------#
#									  #
#      subnet       						          #
#									  #
#          Calculate the subnet that a host is on.                        #
#      INPUT:                                                             #
#	   host    : hostname or ip address                               #
#	   netmask : subnet mask in dotted decimal format (a.b.c.d)       #
#      OUTPUT:                                                            #
#	   returns the subnet in dotted decimal format (a,b,c,d)          #
#									  #
#-------------------------------------------------------------------------#
sub subnet
{
    my ($class, $host, $netmask) = @_;

    #$host = `hostname` if (!$host);
    #$netmask = '255.255.255.0' if (!$netmask);

    chomp $host;
    chomp $netmask;

    #print "INPUT:  $host $netmask\n";

    my ($hostname, $ipaddr) = NodeUtils->getHost($host);

    my ($ia,$ib,$ic,$id) = split ('\.', $ipaddr);
    my ($na,$nb,$nc,$nd) = split ('\.', $netmask);

    # Convert to integers so the bitwise and (&) works correctly.
    int $ia; int $ib; int $ic; int $id;
    int $na; int $nb; int $nc; int $nd;

    my $sa = ($ia & $na);
    my $sb = ($ib & $nb);
    my $sc = ($ic & $nc);
    my $sd = ($id & $nd);
    my $subnet = "$sa.$sb.$sc.$sd";


    #print "IPADDR = $ia.$ib.$ic.$id\n";
    #print "NETMSK = $na.$nb.$nc.$nd\n";
    #print "SUBNET = $subnet\n";

    return $subnet;
}

#-------------------------------------------------------------------------#
#									  #
#      determineMS    						          #
#									  #
#	this returns the name of the localhost from a list of node names
#	input: @DestNode
#									  #
#-------------------------------------------------------------------------#
sub determineMS
{
	my ($class, $refnodes) = @_;
	my @nodes = @$refnodes;
	my ($localnode, $ifconfig, $label, $separator, $address, @ipaddrs, %localhostnames);
	
	if($::PLTFRM eq "AIX"){
		$ifconfig = "/usr/sbin/ifconfig -a";
		# format is: "inet xxx.xxx.xxx.xxx
		$label = "inet";
		$separator = " ";
	}
	else{ #Linux
		$ifconfig = "/sbin/ifconfig -a";
		# format is: "inet addr:xxx.xxx.xxx.xxx
		$label = "inet addr";
		$separator = ":";
	}
	
	NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','V', 'IMsgCMD', $ifconfig);
	my @ips = `$ifconfig 2>&1`;
	my $rc  = $? >> 8;
	if ($rc) {
		NodeUtils->messageFromCat('csmInstall.cat',$::MSGMAPPATH,'csminstall','E', 'EMsgCANT_RUN', $ifconfig, $rc);
	}
	foreach(@ips){
		if ($_ =~ /$label$separator((\d{1,3}?\.){3}(\d){1,3})\s/o) {
			#print "found \"$1\" in \"$_\"\n";
			push @ipaddrs, $1;
		}
	}
	foreach my $addr (@ipaddrs){
		#see what it resolves to
		$localhostnames{NodeUtils->tryHost($addr)}=1;
	}
	FINDMS: foreach my $name (@nodes){
		my $rn = NodeUtils->tryHost($name);
		if($localhostnames{$rn}){
			#there should only be one ManagedNode that maps to the localhost
			$localnode = $name;
			last FINDMS;
		}
	}
	return $localnode;	
}

#-------------------------------------------------------------------------#
#									  #
#      returnSubdirectory    					          #
#									  #
#	this returns all the sub directories (and grandchildren directories, etc)
#       of a give directory
#	input: directory name
#									  #
#-------------------------------------------------------------------------#
sub returnSubdirectory{
	my ($class, $dir, $valid_dir) = @_;
	my $file;
	my @subdirs = ();
	my @returndirs = ();
	my @valid_subdirs = (); #these are ones whose children we can look at	
	if($dir =~ m/(installp.?\z)|(usr.?\z)|(RPMS.?\z)/){ #check to see if the parent is valid
		$valid_dir=1;
	}
	opendir(DIRECTORY, $dir) or return();
	while ($file = readdir(DIRECTORY)){
		if ($file ne '.' and $file ne '..'){
			my $filename;
			if($dir =~ /\/\z/){ #if it ends with a slash
				$filename = "$dir$file";
			}
			else{ #no slash
				$filename = "$dir/$file";
			}
			if(-d "$filename"){
				if(!$valid_dir){ #the parent wasn't an valid directory
					if(($file eq "installp") || ($file eq "usr") || ($file eq "RPMS")){
						push @valid_subdirs, $filename;
					}
				}
				else{ #a valid parent, so therefore its children are valid too
					push @valid_subdirs, $filename;		
				}
				
			}
		}
	}
	closedir(DIRECTORY);
	push @returndirs, @valid_subdirs;
	push @returndirs, @subdirs;
	foreach my $sub (@valid_subdirs){  #these are valid children
		my @tmpdirs = InstallUtils->returnSubdirectory($sub, 1);
		push @returndirs, @tmpdirs;
        }
        return @returndirs;
}

#-------------------------------------------------------------------------#
#									  #
#      getUpdateDir    						          #
#									  #
#	this returns the updates directory corresponding to the 
#	Install attributes that it is passed
#
#	input: (InstallOSName, InstallDistributionName, 
#		InstallDistributionVersion, InstallPkgArchitecture);
#									  #
#-------------------------------------------------------------------------#

sub getUpdateDir{
	my ($class, $os, $distributor, $version, $arch) = @_;
	return "/csminstall/$os/$distributor/$version/$arch/updates";
}

#-------------------------------------------------------------------------#
#									  #
#      getRPMSDir    						          #
#									  #
#	this returns the updates directory corresponding to the 
#	Install attributes that it is passed
#
#	input: (InstallOSName, InstallDistributionName, 
#		InstallDistributionVersion, InstallPkgArchitecture);
#									  #
#-------------------------------------------------------------------------#

sub getRPMSDir{
	my ($class, $os, $distributor, $version, $arch) = @_;
	return "/csminstall/$os/$distributor/$version/$arch/RPMS";
}

#-------------------------------------------------------------------------#
#									  #
#      getBaseDir    						          #
#									  #
#	this returns the updates directory corresponding to the 
#	Install attributes that it is passed
#
#	input: (InstallOSName, InstallDistributionName, 
#		InstallDistributionVersion, InstallPkgArchitecture);
#									  #
#-------------------------------------------------------------------------#

sub getBaseDir{
	my ($class, $os, $distributor, $version, $arch) = @_;
	return "/csminstall/$os/$distributor/$version/$arch";
}

1;
